/******************************************************************************
 * Spine Runtimes License Agreement
 * Last updated April 5, 2025. Replaces all prior versions.
 *
 * Copyright (c) 2013-2025, Esoteric Software LLC
 *
 * Integration of the Spine Runtimes into software or otherwise creating
 * derivative works of the Spine Runtimes is permitted under the terms and
 * conditions of Section 2 of the Spine Editor License Agreement:
 * http://esotericsoftware.com/spine-editor-license
 *
 * Otherwise, it is permitted to integrate the Spine Runtimes into software
 * or otherwise create derivative works of the Spine Runtimes (collectively,
 * "Products"), provided that each user of the Products must obtain their own
 * Spine Editor license and redistribution of the Products in any form must
 * include this license and copyright notice.
 *
 * THE SPINE RUNTIMES ARE PROVIDED BY ESOTERIC SOFTWARE LLC "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL ESOTERIC SOFTWARE LLC BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES,
 * BUSINESS INTERRUPTION, OR LOSS OF USE, DATA, OR PROFITS) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THE SPINE RUNTIMES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *****************************************************************************/

package spine;

#if flash
typedef Function = Dynamic;
#else
typedef Function = haxe.Constraints.Function;
#end

@:generic class Pool<T> {
	private var items:Array<T>;
	private var instantiator:Function;

	public function new(instantiator:Void->T) {
		this.items = new Array<T>();
		this.instantiator = instantiator;
	}

	/**
	 * This pool keeps a reference to the obtained instance, so it should be returned to the pool via Pool.free(),
	 * Pool.freeAll() or clear() to avoid leaking memory.
	 * @return The obtained instance from the pool or a new instance if the pool is empty.
	 */
	public function obtain():T {
		return this.items.length > 0 ? this.items.pop() : this.instantiator();
	}

	public function free(item:T):Void {
		if (Std.isOfType(item, Poolable))
			cast(item, Poolable).reset();
		items.push(item);
	}

	public function freeAll(items:Array<T>):Void {
		for (item in items) {
			free(item);
		}
	}

	public function clear():Void {
		items.resize(0);
	}
}
