/******************************************************************************
 * Spine Runtimes License Agreement
 * Last updated January 1, 2020. Replaces all prior versions.
 *
 * Copyright (c) 2013-2020, Esoteric Software LLC
 *
 * Integration of the Spine Runtimes into software or otherwise creating
 * derivative works of the Spine Runtimes is permitted under the terms and
 * conditions of Section 2 of the Spine Editor License Agreement:
 * http://esotericsoftware.com/spine-editor-license
 *
 * Otherwise, it is permitted to integrate the Spine Runtimes into software
 * or otherwise create derivative works of the Spine Runtimes (collectively,
 * "Products"), provided that each user of the Products must obtain their own
 * Spine Editor license and redistribution of the Products in any form must
 * include this license and copyright notice.
 *
 * THE SPINE RUNTIMES ARE PROVIDED BY ESOTERIC SOFTWARE LLC "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL ESOTERIC SOFTWARE LLC BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES,
 * BUSINESS INTERRUPTION, OR LOSS OF USE, DATA, OR PROFITS) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THE SPINE RUNTIMES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *****************************************************************************/

#if UNITY_2018_1_OR_NEWER
#define PLAYABLE_DIRECTOR_HAS_STOPPED_EVENT
#endif

using Spine.Unity;
using System;
using UnityEngine;
using UnityEngine.Playables;
using UnityEngine.Timeline;

namespace Spine.Unity.Playables {
	public class SpineSkeletonFlipMixerBehaviour : PlayableBehaviour {
		float originalScaleX, originalScaleY;
		float baseScaleX, baseScaleY;

		SpinePlayableHandleBase playableHandle;
		bool m_FirstFrameHappened;

#if PLAYABLE_DIRECTOR_HAS_STOPPED_EVENT
		PlayableDirector director = null;

		public override void OnPlayableCreate (Playable playable) {
			director = playable.GetGraph().GetResolver() as PlayableDirector;
			if (director)
				director.stopped += OnDirectorStopped;
		}

		public override void OnPlayableDestroy (Playable playable) {
			if (director)
				director.stopped -= OnDirectorStopped;

			base.OnPlayableDestroy(playable);
		}

		void OnDirectorStopped (PlayableDirector obj) {
			OnStop();
		}
#else
		public override void OnGraphStop (Playable playable) {
			OnStop();
		}
#endif

		public override void ProcessFrame (Playable playable, FrameData info, object playerData) {
			playableHandle = playerData as SpinePlayableHandleBase;

			if (playableHandle == null)
				return;

			var skeleton = playableHandle.Skeleton;

			if (!m_FirstFrameHappened) {
				originalScaleX = skeleton.ScaleX;
				originalScaleY = skeleton.ScaleY;
				baseScaleX = Mathf.Abs(originalScaleX);
				baseScaleY = Mathf.Abs(originalScaleY);
				m_FirstFrameHappened = true;
			}

			int inputCount = playable.GetInputCount();

			float totalWeight = 0f;
			float greatestWeight = 0f;
			int currentInputs = 0;

			for (int i = 0; i < inputCount; i++) {
				float inputWeight = playable.GetInputWeight(i);
				ScriptPlayable<SpineSkeletonFlipBehaviour> inputPlayable = (ScriptPlayable<SpineSkeletonFlipBehaviour>)playable.GetInput(i);
				SpineSkeletonFlipBehaviour input = inputPlayable.GetBehaviour();

				totalWeight += inputWeight;

				if (inputWeight > greatestWeight) {
					SetSkeletonScaleFromFlip(skeleton, input.flipX, input.flipY);
					greatestWeight = inputWeight;
				}

				if (!Mathf.Approximately(inputWeight, 0f))
					currentInputs++;
			}

			if (currentInputs != 1 && 1f - totalWeight > greatestWeight) {
				skeleton.ScaleX = originalScaleX;
				skeleton.ScaleY = originalScaleY;
			}
		}

		public void SetSkeletonScaleFromFlip (Skeleton skeleton, bool flipX, bool flipY) {
			skeleton.ScaleX = flipX ? -baseScaleX : baseScaleX;
			skeleton.ScaleY = flipY ? -baseScaleY : baseScaleY;
		}

		public void OnStop () {
			m_FirstFrameHappened = false;

			if (playableHandle == null)
				return;

			var skeleton = playableHandle.Skeleton;
			skeleton.ScaleX = originalScaleX;
			skeleton.ScaleY = originalScaleY;
		}
	}

}
